
// ======= Config & Storage Keys =======
const DEFAULT_BASE = "http://45.252.248.148:8010";
const STORE_KEYS = {
  BASE: "base_origin",
  USER: "auth_user",
  XTOKEN: "x_token",
  DEVICES: "devices",        // [{name, device_token}]
  AUTOCALL: "auto_call",     // bool
  PANEL: "panel_cfg"
};


// ======= Call quota per SIM/day =======
const QUOTA_KEYS = {
  LIMIT: "quota_limit_per_sim",      // number, default 250
  DATE:  "quota_date",               // yyyy-mm-dd
  C1:    "quota_count_sim1",
  C2:    "quota_count_sim2"
};

async function getTodayStr(){
  const d = new Date();
  const mm = String(d.getMonth()+1).padStart(2,'0');
  const dd = String(d.getDate()).padStart(2,'0');
  return `${d.getFullYear()}-${mm}-${dd}`;
}

async function readQuota(){
  const today = await getTodayStr();
  const st = await chrome.storage.sync.get({
    [QUOTA_KEYS.LIMIT]: 4000,
    [QUOTA_KEYS.DATE]: today,
    [QUOTA_KEYS.C1]: 0,
    [QUOTA_KEYS.C2]: 0
  });
  // reset if date changed
  if (st[QUOTA_KEYS.DATE] !== today){
    st[QUOTA_KEYS.DATE] = today;
    st[QUOTA_KEYS.C1] = 0;
    st[QUOTA_KEYS.C2] = 0;
    await chrome.storage.sync.set(st);
  }
  return st;
}

async function incQuota(sim){
  const st = await readQuota();
  const key = (Number(sim)===2) ? QUOTA_KEYS.C2 : QUOTA_KEYS.C1;
  st[key] = (st[key]||0) + 1;
  await chrome.storage.sync.set({[key]: st[key]});
  return st[key];
}

async function canCall(sim){
  const st = await readQuota();
  const cnt = (Number(sim)===2) ? (st[QUOTA_KEYS.C2]||0) : (st[QUOTA_KEYS.C1]||0);
  const limit = Number(st[QUOTA_KEYS.LIMIT]||250);
  return {ok: cnt < limit, count: cnt, limit};
}

async function resetQuota(){
  const today = await getTodayStr();
  await chrome.storage.sync.set({
    [QUOTA_KEYS.DATE]: today,
    [QUOTA_KEYS.C1]: 0,
    [QUOTA_KEYS.C2]: 0
  });
}


// ======= Utils =======

function buildAuthHeaders(tok){
  const h = {"Content-Type":"application/json"};
  if (tok){ h["X-Token"] = tok; h["Authorization"] = "Bearer " + tok; }
  return h;
}

function sleep(ms){ return new Promise(r => setTimeout(r, ms)); }
function cleanNumber(raw){ return String(raw||"").replace(/[^\d+]/g, ""); }
function toVN(raw){
  const s = cleanNumber(raw);
  if (s.startsWith("+84") && s.length>=12) return "0" + s.slice(-9);
  if (s.startsWith("0084") && s.length>=13) return "0" + s.slice(-9);
  if (/^0\d{9}$/.test(s)) return s;
  return s;
}
async function getCfg(){
  const cfg = await chrome.storage.sync.get({
    [STORE_KEYS.BASE]: DEFAULT_BASE,
    [STORE_KEYS.USER]: "",
    [STORE_KEYS.XTOKEN]: "",
    [STORE_KEYS.DEVICES]: [],
    [STORE_KEYS.AUTOCALL]: false,
    [STORE_KEYS.PANEL]: {opacity:0.96, z:2147483647, btn_w:116, btn_h:40, show_on_new:true, auto_one_sim:false},
  });
  return cfg;
}
function buildOrigin(baseLike){
  try { return new URL(baseLike).origin; }
  catch (e) {
    return String(baseLike||"").replace(/\/+$/,"");
  }
}
async function apiPost(pathOrUrl, body){
  const cfg = await getCfg();
  const origin = buildOrigin(cfg[STORE_KEYS.BASE] || DEFAULT_BASE);
  const url = pathOrUrl.startsWith("http") ? pathOrUrl : origin + pathOrUrl;
  const headers = {"Content-Type":"application/json"};
  const tok = (cfg[STORE_KEYS.XTOKEN]||"").trim();
  if (tok) headers["X-Token"] = tok;
  const res = await fetch(url, {method:"POST", headers, body: JSON.stringify(body||{})});
  if (!res.ok){
    let t = await res.text().catch(()=> "");
    throw new Error(`HTTP ${res.status}: ${t||res.statusText}`);
  }
  try { return await res.json(); } catch(_){ return await res.text(); }
}

// Try multiple endpoints for each action (robust to server paths)



async function callNumber({device_token, number, sim}){
  const cfg = await getCfg();
  const base = buildOrigin(cfg[STORE_KEYS.BASE] || DEFAULT_BASE);
  const user = cfg[STORE_KEYS.USER] || "";
  const tok  = (cfg[STORE_KEYS.XTOKEN]||"").trim();
  const headers = {"Content-Type":"application/json"};
  if (tok) headers["X-Token"] = tok;

  const cleanNum = toVN(number);
  const withSimBody = { device_token, number: cleanNum, sim: (Number(sim)||1), autoCall: (!!cfg[STORE_KEYS.AUTOCALL]) ? "true" : "false" };
  const origin = base.replace(/\/+$/,"");
  const urls = [
    `${origin}/api/call/dial?user=${encodeURIComponent(user)}`,
    `${origin}/call/dial?user=${encodeURIComponent(user)}`,
    `${origin}/api/calls/dial?user=${encodeURIComponent(user)}`,
    `${origin}/calls/dial?user=${encodeURIComponent(user)}`,
    `${origin}/devices/call`,
    `${origin}/api/devices/call`,
    `${origin}/call/dial`,
    `${origin}/api/call/dial`,
    `${origin}/calls/dial`,
    `${origin}/api/calls/dial`
  ];
  let lastErr;
  for (const u of urls){
    try{
      const r = await fetch(u, {method:"POST", headers, body: JSON.stringify(withSimBody)});
      if (r.ok) return await r.text();
      lastErr = new Error(`HTTP ${r.status}: ${await r.text()}`);
    }catch(e){ lastErr = e; }
  }
  throw lastErr || new Error("No working call endpoint");

}


// Test whether a device token is reachable/active (non-intrusive)
// Strategy: prefer harmless actions (ping/echo/clipboard) over call.
// Success if any endpoint returns 2xx.
/* duplicate testDevice removed during patch */
async function testDevice({device_token}){
  const cfg = await getCfg();
  const base = buildOrigin(cfg[STORE_KEYS.BASE] || DEFAULT_BASE);
  const user = cfg[STORE_KEYS.USER] || "";
  const tok  = (cfg[STORE_KEYS.XTOKEN]||"").trim();
  const headers = {"Content-Type":"application/json"};
  if (tok) headers["X-Token"] = tok;

  const b = base.replace(/\/+$/,"");
  const variants = [
    `${b}/api/call/dial?user=${encodeURIComponent(user)}`,
    `${b}/call/dial?user=${encodeURIComponent(user)}`,
    `${b}/api/calls/dial?user=${encodeURIComponent(user)}`,
    `${b}/calls/dial?user=${encodeURIComponent(user)}`
  ];

  // minimal body to avoid real call; expect 4xx (400/422) as "reachable"
  const body = JSON.stringify({ device_token });

  let lastErr;
  for (const u of variants){
    try{
      const r = await fetch(u, {method:"POST", headers, body});
      if (r.status === 404){ lastErr = new Error("404 Not Found"); continue; }
      if (r.status === 401 || r.status === 403){ lastErr = new Error("401/403 Unauthorized"); continue; }
      // 2xx or 4xx other than 401/403/404 => endpoint reachable
      return {ok:true, url:u, code:r.status, text: await r.text()};
    }catch(e){ lastErr = e; }
  }
  throw lastErr || new Error("No working test endpoint");
}

// ======= Message Router from content/popup =======
chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  (async () => {
    try {
      if (msg && msg.type === "ccp_call"){
        const {sim} = msg.payload || {};
        const qc = await canCall(sim);
        if (!qc.ok){ sendResponse({ok:false, error:`Vượt giới hạn gọi cho SIM${Number(sim)===2?2:1}: ${qc.count}/${qc.limit}`}); return; }
        const r = await callNumber(msg.payload);
        await incQuota(sim);
        try{ chrome.runtime.sendMessage({type:"ccp_quota_updated"}); }catch(_){}
        sendResponse({ok:true, data:r}); return;
      }
      if (msg && msg.type === "ccp_hangup"){
        const r = await hangup(msg.payload);
        sendResponse({ok:true, data:r}); return;
      }
      if (msg && msg.type === "ccp_paste"){
        const r = await pasteNumber(msg.payload);
        sendResponse({ok:true, data:r}); return;
      }
      if (msg && msg.type === "ccp_test_device"){ const r = await testDevice(msg.payload); sendResponse({ok:true, data:r}); return; }
      if (msg && msg.type === "ccp_test_device"){ const r = await testDevice(msg.payload); sendResponse({ok:true, data:r}); return; }

      if (msg && msg.type === "ccp_get_quota"){
        const st = await readQuota();
        sendResponse({ok:true, data: st}); return;
      }
      if (msg && msg.type === "ccp_reset_quota"){
        await resetQuota();
        const st = await readQuota();
        sendResponse({ok:true, data: st}); return;
      }
      if (msg && msg.type === "ccp_set_quota_limit"){
        const lim = Math.max(1, Number(msg.limit||400));
        await chrome.storage.sync.set({[QUOTA_KEYS.LIMIT]: lim});
        const st = await readQuota();
        sendResponse({ok:true, data: st}); return;
      }
      if (msg && msg.type === "ccp_get_cfg"){
        sendResponse({ok:true, data: await getCfg()}); return;
      }
      if (msg && msg.type === "ccp_toggle_panel"){
        // toggle flag in storage so content.js can draw/hide
        const st = await chrome.storage.local.get({panel_enabled:false});
        const next = !st.panel_enabled;
        await chrome.storage.local.set({panel_enabled: next});
        sendResponse({ok:true, enabled: next}); return;
      }
    } catch(e){
      sendResponse({ok:false, error: e.message || String(e)});
    }
  })();
  return true; // keep channel open for async
});

// Toolbar click -> toggle panel

chrome.action.onClicked.addListener(async (tab) => {
  if (!tab || !tab.id) return;
  try { await chrome.tabs.sendMessage(tab.id, {type:"ccp_toggle_panel"}); return; } catch(_){ }
  try { await chrome.scripting.executeScript({target:{tabId:tab.id}, files:["content.js"]}); await chrome.tabs.sendMessage(tab.id, {type:"ccp_show_panel"}); return; } catch(_){ }
  try{
    if (!tab || !tab.id) return;
    try{
      await chrome.tabs.sendMessage(tab.id, {type:"ccp_toggle_panel"});
      return;
    }catch(_){}
    // Fallback: inject content.js then toggle
    await chrome.scripting.executeScript({target:{tabId:tab.id}, files:["content.js"]});
    await chrome.tabs.sendMessage(tab.id, {type:"ccp_toggle_panel"});
  }catch(e){
    console.warn("Toggle failed:", e);
  }
});


chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  (async () => {
    try{
      if (msg && msg.type === "ccp_broadcast_panel_cfg"){
        // Save too (keep source of truth in sync storage)
        const pcfg = msg.payload || {};
        try{
          await chrome.storage.sync.set({["panel_cfg"]: pcfg});
        }catch(_){}
        // Send to all tabs that have our content script
        const tabs = await chrome.tabs.query({});
        for (const t of tabs){
          try{
            await chrome.tabs.sendMessage(t.id, {type:"ccp_apply_panel_cfg", payload: pcfg});
          }catch(_){}
        }
        sendResponse({ok:true});
        return;
      }
    }catch(e){
      sendResponse({ok:false, error: e.message || String(e)});
    }
  })();
  return true;
});
